/* 
   CC0 2011, Martin Haye

   To the extent possible under law, Martin Haye has waived all copyright 
   and related or neighboring rights to p2e: Pseudo-II Emulator. 
   This work is published from: United States.
*/

/* Support for slow but sure checking of the emulation */

function Checker()
{
  var outerState = new Object();
  var innerState = new Object();
  
  var CxxxMode, lowerMode, upperMode, displayPage;
  var DxxxMode, ExxxMode, lcWriteMode;
  var store80, auxReadMode, auxWriteMode, auxZpMode;
  var addrsWritten;

  var diskDrive = new DiskDrive();
  diskDrive.setQuiet(true);
  
  var triggerTime = -1;
  var triggerAddr = -1;
  var debugCount = 0;
  
  function saveState(state)
  {
    state.raw_mem = raw_mem;
    state.mem_set = mem_set;
    state.mem_get = mem_get;
    state.kbdBuf = kbdBuf;
    state.prevKey = prevKey;
    
    state.a = a;
    state.x = x;
    state.y = y;
    state.v = v;
    state.c = c;
    state.nz = nz;
    state.fd = fd;
    state.fi = fi;
    state.fb = fb;
    state.pc = pc;
    state.s = s;
    state.t = t;
  }
  
  function restoreState(state)
  {
    raw_mem = state.raw_mem;
    mem_set = state.mem_set;
    mem_get = state.mem_get;
    kbdBuf = state.kbdBuf;
    prevKey = state.prevKey;
    
    a = state.a;
    x = state.x;
    y = state.y;
    v = state.v;
    c = state.c;
    nz = state.nz;
    fd = state.fd;
    fi = state.fi;
    fb = state.fb;
    pc = state.pc;
    s = state.s;
    t = state.t;
  }
  
  function debugStr(state, prevT)
  {
    return [toHex(state.pc,4),
            "-   A=",toHex(state.a,2),
            " X=",toHex(state.x,2),
            " Y=",toHex(state.y,2),
            " C=",bool2(state.c),
            " N=",bool2((state.nz & 0x80) ? 1 : 0),
            " Z=",bool2((state.nz & 0x200) || !state.nz),
            " nz=",toHex(nz,2),
            " V=",bool2(state.v),
            " S=",toHex(state.s,2),
            " T=",t,
            ((prevT !== undefined) ? (" dT=" + (t-prevT)) : ""),
            " stk=[", toHex(state.raw_mem[state.s+0x101],2), " ", 
                      toHex(state.raw_mem[state.s+0x102],2), " ",
                      toHex(state.raw_mem[state.s+0x103],2), " ",
                      toHex(state.raw_mem[state.s+0x104],2), "]",
            " ",opStrings[mem_get(state.pc)],
           ].join("");
  }
  
  function checkStates()
  {
    if (innerState.pc != outerState.pc ||
        innerState.t != outerState.t ||
        innerState.a != outerState.a ||
        innerState.x != outerState.x ||
        innerState.y != outerState.y ||
        innerState.v != outerState.v ||
        innerState.c != outerState.c ||
        innerState.nz != outerState.nz ||
        innerState.fd != outerState.fd ||
        innerState.fi != outerState.fi ||
        innerState.fb != outerState.fb ||
        innerState.s != outerState.s)
    {
      console.debug("Mismatch at t=" + innerState.t + "/" + outerState.t);
      console.debug("Inner: " + debugStr(innerState));
      console.debug("Outer: " + debugStr(outerState));
      assert(false);
    }
    
    for (var i in addrsWritten) {
      var addr = addrsWritten[i];
      assert(innerState.raw_mem[addr] == outerState.raw_mem[addr]);
    }
    
    assert(innerState.raw_mem[11] == outerState.raw_mem[11]);
    assert(innerState.raw_mem[257] == outerState.raw_mem[257]);

    //for (var i=0; i<=0xFFFF; i++)
    //  assert(innerState.raw_mem[i] == outerState.raw_mem[i]);
  }

  function special_get(addr)
  {
    if (addr >= 0xC0E0 && addr < 0xC0EF) // Disk II
      return diskDrive.softswitch_get(addr);
      
    switch (addr)
    {
      case 0xC000: // read keyboard
        if (kbdBuf != "")
          prevKey = (kbdBuf.charCodeAt(0) & 0xFF) | 0x80;
        return prevKey;

      case 0xC010: // strobe keyboard
        prevKey &= 0x7F;
        if (kbdBuf != "")
          kbdBuf = kbdBuf.substr(1);
        return 0;
    
      case 0xC011: // reading from LC bank 2?
        return DxxxMode == "ram2" ? 128 : 0;
        
      case 0xC012: // reading from LC ram?
        return ExxxMode == "ram" ? 128 : 0;
        
      case 0xC015: // RDCXROM
        return CxxxMode == "ROM" ? 128 : 0;
    
      case 0xC01A: // RDTEXT
        return upperMode == "text" ? 128 : 0;
    
      case 0xC01B: // RDMIXED
        return upperMode != lowerMode ? 128 : 0;
        
      case 0xC01C: // RDPAGE2
        return displayPage ? 128 : 0;
        
      case 0xC01D: // RDHIRES
        return upperMode == "hgr" ? 128 : 0;

      case 0xC080:
      case 0xC084:
        DxxxMode = "ram2";
        ExxxMode = "ram";
        lcWriteMode = "protect";
        return 0;
        
      case 0xC081:
      case 0xC085:
        DxxxMode = "rom";
        ExxxMode = "rom";
        lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write2" : "protect2";
        return 0;
        
      case 0xC082:
      case 0xC086:
        DxxxMode = "rom";
        ExxxMode = "rom";
        lcWriteMode = "protect";
        return 0;
        
      case 0xC083:
      case 0xC087:
        DxxxMode = "ram2";
        ExxxMode = "ram";
        lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write2" : "protect2";
        return 0;
        
      case 0xC088:
      case 0xC08C:
        DxxxMode = "ram1";
        ExxxMode = "ram";
        lcWriteMode = "protect";
        return 0;
        
      case 0xC089:
      case 0xC08D:
        DxxxMode = "rom";
        ExxxMode = "rom";
        lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write1" : "protect2";
        return 0;
        
      case 0xC08A:
      case 0xC08E:
        DxxxMode = "rom";
        ExxxMode = "rom";
        lcWriteMode = "protect";
        return 0;
        
      case 0xC08B:
      case 0xC08F:
        DxxxMode = "ram1";
        ExxxMode = "ram";
        lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write1" : "protect2";
        return 0;
        
      default:
        // No match, zero seems a safe general return value.
        return 0;
    }
  }
  
  function getter(addr)
  {
    if ((addr < 0x200 || addr >= 0xD000) && auxZpMode == "aux")
      return 0;
    if (addr >= 0x200 && addr < 0xC000 && auxReadMode == "aux")
      return 0;
    if (addr < 0xC000)
      return raw_mem[addr];
    if (addr < 0xC100)
      return special_get(addr);
    if (addr >= 0xC300 && addr <= 0xC3FF)
      return rom_data[CXXX_ROM_OFFSET + (addr - 0xC000)];
    if (addr < 0xD000) {
      if (CxxxMode == "card")
        return rom_data[CXXX_CARD_OFFSET + (addr - 0xC000)];
      else 
        return rom_data[CXXX_ROM_OFFSET + (addr - 0xC000)];
    }
    if (addr < 0xE000) {
      if (DxxxMode == "ram1")
        return raw_mem[addr - 0x1000];
      else if (DxxxMode == "ram2")
        return raw_mem[addr];
      else
        return rom_data[BASIC_ROM_OFFSET + (addr - 0xD000)];
    }
    if (ExxxMode == "ram")
      return raw_mem[addr];
    return rom_data[BASIC_ROM_OFFSET + (addr - 0xD000)];
  }
  
  function setter(addr, val)
  {
    getter(addr); // simulate two gets per 6502
    getter(addr); // simulate two gets per 6502

    addrsWritten.push(addr);
    
    if ((addr < 0x200 || addr >= 0xD000) && auxZpMode == "aux")
      return;
    if ((addr >= 0x400 && addr < 0x800) && store80 == "aux")
      return;
    if (addr >= 0x200 && addr < 0xC000 && auxWriteMode == "aux")
      return;
    
    if (addr < 0xC000) {
      raw_mem[addr] = val;
      return;
    }
    if (addr < 0xC100) {
      switch (addr)
      {
        case 0xC000: // CLR80STORE
          store80 = "main";
          break;
        case 0xC001: // SET80STORE
          store80 = "aux";
          break;
        case 0xC002: // CLRAUXRD
          auxReadMode = "main";
          break;
        case 0xC003: // SETAUXRD
          auxReadMode = "aux";
          break;
        case 0xC004: // CLRAUXWR
          auxWriteMode = "main";
          break;
        case 0xC005: // SETAUXWR
          auxWriteMode = "aux";
          break;
        case 0xC006: // CLRCXROM
          CxxxMode = "card";
          break;
        case 0xC007: // SETCXROM
          CxxxMode = "ROM";
          break;
        case 0xC008: // CLRAUXZP
          auxZpMode = "main";
          break;
        case 0xC009: // SETAUXZP
          auxZpMode = "aux";
          break;
      }
      return;
    }
    if (addr >= 0xD000 && addr <= 0xDFFF) {
      if (lcWriteMode == "write1") {
        raw_mem[addr-0x1000] = val;
        return;
      }
      if (lcWriteMode == "write2") {
        raw_mem[addr] = val;
        return;
      }
    }
    if (addr >= 0xE000 && (lcWriteMode == "write1" || lcWriteMode == "write2")) {
      raw_mem[addr] = val;
      return;
    }
    
    // Ignore write to ROM
    return;
  }
  
  this.addKey = function(key)
  {
    innerState.kbdBuf += key;
  }
  
  // One-time initialization
  this.init = function(new_pc)
  {
    saveState(outerState);
    raw_mem = new Array();
    mem_set = new Array();
    mem_get = getter;
    for (var i=0; i<=0xFFFF; i++) {
      raw_mem[i] = 0;
      mem_set[i] = setter;
    }
    a = x = y = c = v = nz = t = 0;
    s = 0xFF;
    pc = new_pc;
    saveState(innerState);
    restoreState(outerState);
    
    // Non-saved stuff
    upperMode = lowerMode = "text";
    CxxxMode = "card";
    DxxxMode = "rom";
    ExxxMode = "rom";
    auxZpMode = "main";
    auxReadMode = "main";
    auxWriteMode = "main";
    store80 = "main";
    displayPage = 0;
  }
  
  this.insertDisk = function(diskData) {
    diskDrive.insert(diskData);
  }
  
  this.setTriggerTime = function(t) {
    triggerTime = t;
  }
  
  this.setTriggerAddr = function(t) {
    triggerAddr = t;
  }
  
  // Main entry point
  this.runCycles = function(end)
  {
    saveState(outerState);
    restoreState(innerState);
    
    addrsWritten = new Array();
    
    // Sanity check
    if (mem_get(pc) == mem_get(pc+1) && mem_get(pc) == mem_get(pc+2) &&
        (mem_get(pc) == 0xA0 || mem_get(pc) == 0))
      assert(false, "probably executing bad memory area");
    
    var prevT = t;
    while (t < end) 
    {
      if ((triggerTime >= 0 && t >= triggerTime && t < triggerTime + 100))
      {
        debugCount = 200;
        triggerTime = -1;
      }
      else if (pc == triggerAddr) {
        debugCount = 50;
        triggerAddr = -1;
      }
      
      if (debugCount > 0) {
        saveState(innerState);
        console.debug("i" + debugStr(innerState, prevT));
        debugCount--;
        if (debugCount == 0)
          assert(false); // no reason to go further, we're debugging.
      }
      prevT = t;
      opTbl[mem_get(pc++)]();
    }
    saveState(innerState);
    restoreState(outerState);
    checkStates();
  }
}
